/*-------------------------------------------------------*
*  Name:      PKG filter for ZModeler v1.06              *
*  Purpose:   Midown Madness 2 files import/export       *
*  Authors:   Oleg M.                                    *
*                                                        *
*  History:   16.Sep.2001 - rebuild for ZModeler v1.05   *
*             24.Mar.2002 - rebuild for ZModeler v1.06   *
*                                                        *
*-------------------------------------------------------*/
#include <Struct.h>
#include <3DEngine.h>


#include "resource.h"
#include "Export.h"
#include "Pkg.h"
#include "VertList.h"
#include "Lists.h"
#include "Var.h"


//////////////////////////////////////////////////////////////
//  DWORD Capabilities - a combination versions
DWORD CALLBACK Capabilities(long)
{
  return ZMODELER_VERSION;
}

//////////////////////////////////////////////////////////////
//  DWORD Supports returns a combination of supported features
DWORD CALLBACK Supports(DWORD)
{
  return Z3D_PLUGSUPPORTS_IMPORT | Z3D_PLUGSUPPORTS_EXPORT;
}

//////////////////////////////////////////////////////////////
//  DWORD DynamicLoading specifies whether the dll will be
//  loaded only when it'sfunctions are needed.
//  You should NOT make it dynamic, if it is a processor-type.
DWORD CALLBACK DynamicLoading(DWORD)
{
  return 1;
}

//////////////////////////////////////////////////////////////
//  void GetFilterExtention returns a file extention for sup-
//  ported files. Used if it is an IMPORT/EXPORT type
char* CALLBACK GetFilterExtension(DWORD)
{
  return "pkg";
}

//////////////////////////////////////////////////////////////
//  void GetFilterMask returns a file-mask for CFileDilaog.
//  Used if it is an IMPORT/EXPORT type.
char* CALLBACK GetFilterMask(DWORD)
{
  return "Midtown Madness 2 (*.pkg)|*.pkg|";
}




//////////////////////////////////////////////////////////////
//  Shows a failure message.
long _stdcall ShowFailMessage(CWnd* pwnd, char *errmsg, char *cause, UINT icon)
{
  AfxGetApp()->DoWaitCursor(0);
  char str[512];
  strcpy(str, "Error importing/exporting file: ");
  strcat(str, errmsg);
  strcat(str, "\nPossible caused by: ");
  strcat(str, cause);
  long res = pwnd->MessageBox(str,"Filter message:", icon);
  if (res == IDOK) return 0;//default return;

  return res; //custom return (Yes/No/Cancel);
}



/**********************************************************************/
//    MM2 PKG File Importer for Zanoza 3D Editor
/**********************************************************************/
DWORD CALLBACK Import(  CString fromfile,
              CWnd *pwnd,
            tObjectSet* Objects,
            tUnDataSet* UnData,
            CDirect3D*  d3d,
            SYSTEMREQUESTPROC RequestProc,
            HINSTANCE AppHIns,
            HINSTANCE DllHIns)
{
  BOOL  LogEnabed = FALSE;
  AfxGetApp()->DoWaitCursor(1);

  CFile InFile;
  
  if (!InFile.Open(fromfile, CFile::modeRead))
    return ShowFailMessage(  pwnd, "Error opening input file for importing",
                "already opened, or has invalid attributes",
                MB_OK | MB_ICONEXCLAMATION);


  CString RealFileName = fromfile.Left(fromfile.GetLength()-4); // - ".pkg"
  int slashpos = RealFileName.ReverseFind('\\');
  if (slashpos == -1) slashpos = RealFileName.ReverseFind('/');
  if (slashpos != -1)
    RealFileName = RealFileName.Right(RealFileName.GetLength()-slashpos-1);

  //Lets Read the file.
  char ID[5]; ID[4] = 0;
  InFile.Read(ID, 4);
  long  FilePosition = 4;
  long  FileSize = 0;
  while (  InFile.Seek(FilePosition, CFile::begin) &&  //seek to new file
      InFile.Read(&sFileID, sizeof(sFileID)) &&  //read prefix 
      (sFileID.idFILE[0] == 'F') &&        //compare with "FILE"
      (sFileID.idFILE[1] == 'I') &&
      (sFileID.idFILE[2] == 'L') &&
      (sFileID.idFILE[3] == 'E'))
  {//readed correctly.
    InFile.Read(&FileName, sFileID.NameLen);
    InFile.Read(&FileSize, 4);
    FilePosition = InFile.GetPosition() + FileSize;

    if (CharMatch((char*)&FileName, "shaders"))
    {//materials
      DWORD UnKnown1, UnKnown2;
      BYTE TextureLen;
      char TextureName[256];
      InFile.Read(&UnKnown1,  4);
      InFile.Read(&UnKnown2,  4);
      InFile.Read(&TextureLen,1);
      BYTE readed = 0;
      while ((InFile.GetPosition() < (unsigned)FilePosition) && (readed < UnKnown2))
      {
        strcpy(TextureName, "Matte material\0");
        InFile.Read(TextureName, TextureLen);
        TextureLen = 0;
        int thistexturematerial = 0;

        // add "..\texture\" search path
        CString TexturePath = fromfile.Left(fromfile.ReverseFind('\\'));
        TexturePath = TexturePath.Left(TexturePath.ReverseFind('\\')) + CString("\\Texture\\");
        d3d->Materials.Textures->AddPath(TexturePath.GetBuffer(64));

        while ((TextureLen == 0) && (InFile.GetPosition() < (unsigned)FilePosition))
        {
          if (UnKnown1 == 9)
          {//dash materials
            InFile.Read(&PKGDashMat, sizeof(sDashMaterial));
            PKGMat.rgba_ambient.r = BYTE(PKGDashMat.rgba_ambient.r*255.0f),
            PKGMat.rgba_ambient.g = BYTE(PKGDashMat.rgba_ambient.g*255.0f),
            PKGMat.rgba_ambient.b = BYTE(PKGDashMat.rgba_ambient.b*255.0f),
            PKGMat.rgba_ambient.a = BYTE(PKGDashMat.rgba_ambient.a*255.0f),
            PKGMat.power = PKGDashMat.power;
          }//dash materials
          else
            InFile.Read(&PKGMat, sizeof(sMaterial));
          d3d->CreateMaterial(
              (float)PKGMat.rgba_ambient.r/255.0f,
              (float)PKGMat.rgba_ambient.g/255.0f,
              (float)PKGMat.rgba_ambient.b/255.0f,
              1.0f,//(float)PKGMat.rgba_ambient.a/255.0f,
              PKGMat.power,
              80.0f,
              CString(CString(TextureName)+CString(".tga")).GetBuffer(64),
              NULL,
              NULL,
              NULL,
              CString(CString(TextureName)+Int2String(thistexturematerial)).GetBuffer(64),
              0,
              PKGMat.rgba_ambient.a > 0 ? D3DTOP_MODULATE : D3DTOP_ADD,
              D3DTOP_DISABLE,
              D3DTOP_ADD,
              D3DTOP_DISABLE,
              D3DBLEND_SRCALPHA,
              D3DBLEND_INVSRCALPHA,
              1,  //glow by default;
              0,
              D3DCMP_ALWAYS,
              0,0,FALSE);
          InFile.Read(&TextureLen,1);
          thistexturematerial++;
          readed++;
        }
      }
    }//materials
    else
    if (CharMatch((char*)&FileName, "offset"))
    {//unknown
    }//unknown
    else
    {//read mesh
      sVertex*  pVertBuffer;
      WORD*    pFaces;
      long    NumFaces;
      InFile.Read(&FileDescriptor, sizeof(sFileDescriptor));
      //create object;
      tObject*  pObject = new tObject;
      strcpy(pObject->ObjectName, (char*)&FileName);
      pObject->VertTable->VertAmount = FileDescriptor.lTotalVerts;
      pObject->VertTable->Table = new tNormalVertex[FileDescriptor.lTotalVerts];
      pObject->FaceTable->FaceAmount = FileDescriptor.lTotalFaces/3;
      pObject->FaceTable->Table = new tFace[FileDescriptor.lTotalFaces/3];
      CurrVerts = 0;
      CurrFaces = 0;

      CString Name;
      int pos = fromfile.ReverseFind('.');
      if (pos > 0)Name = fromfile.Left(pos);
      else    Name = fromfile;
      Name+=CString("_");
      pos = CString(FileName).ReverseFind('_'); //_H _M _L exclude
      if (pos > 0)Name += CString(FileName).Left(pos);
      else    Name += CString(FileName);
      Name+=CString(".mtx");
      
      int i;
      for (i = 0; i < 12; i++) Transform[i] = 0.0f;
      if (MiscFile.Open(  Name, CFile::modeRead))
      {//load Matrix
        for (i = 0; i < 12; i++)
          MiscFile.Read(&Transform[i], 4);
        MiscFile.Close();
        strcat(pObject->ObjectName, ":m");
      }//load Matrix


      while (InFile.GetPosition() < (unsigned)FilePosition)
      {//sections
        //load objects' section header
        InFile.Read(&Section, sizeof(sSection));
        // load vertices
        pVertBuffer = new sVertex[Section.lNumVertices];
        InFile.ReadHuge(pVertBuffer, sizeof(sVertex)*Section.lNumVertices);

        // load Faces
        InFile.Read(&NumFaces,4);
        pFaces = new WORD[NumFaces];
        InFile.ReadHuge(pFaces, 2*NumFaces);

        tPOINT center = tPOINT (-Transform[9], Transform[10], Transform[11]);
        SetMatrixPosition(pObject->mTransform, center);
        // apply;
        for (int vert = 0; vert < Section.lNumVertices; vert++)
        {
          pObject->VertTable->Table[CurrVerts + vert].X = -(pVertBuffer[vert].x + Transform[9]);
          pObject->VertTable->Table[CurrVerts + vert].Y = pVertBuffer[vert].y + Transform[10];
          pObject->VertTable->Table[CurrVerts + vert].Z = -(pVertBuffer[vert].z + Transform[11]);
          pObject->VertTable->Table[CurrVerts + vert].NormalX = -pVertBuffer[vert].nx;
          pObject->VertTable->Table[CurrVerts + vert].NormalY = pVertBuffer[vert].ny;
          pObject->VertTable->Table[CurrVerts + vert].NormalZ = -pVertBuffer[vert].nz;
        }
        for (int face = 0; face < NumFaces/3; face++)
        {
          pObject->FaceTable->Table[CurrFaces + face].I2 = CurrVerts + pFaces[face*3];
          pObject->FaceTable->Table[CurrFaces + face].I1 = CurrVerts + pFaces[face*3+1];
          pObject->FaceTable->Table[CurrFaces + face].I3 = CurrVerts + pFaces[face*3+2];
          pObject->FaceTable->Table[CurrFaces + face].U1 = pVertBuffer[pFaces[face*3+1]].u;
          pObject->FaceTable->Table[CurrFaces + face].V1 = pVertBuffer[pFaces[face*3+1]].v;
          pObject->FaceTable->Table[CurrFaces + face].U2 = pVertBuffer[pFaces[face*3]].u;
          pObject->FaceTable->Table[CurrFaces + face].V2 = pVertBuffer[pFaces[face*3]].v;
          pObject->FaceTable->Table[CurrFaces + face].U3 = pVertBuffer[pFaces[face*3+2]].u;
          pObject->FaceTable->Table[CurrFaces + face].V3 = pVertBuffer[pFaces[face*3+2]].v;
          pObject->FaceTable->Table[CurrFaces + face].Material = d3d->Materials.MaterialsAmount+Section.Material;
        }
        CurrVerts += Section.lNumVertices;
        CurrFaces += NumFaces/3;

        // delete
        delete[]  pVertBuffer;
        delete[]  pFaces;
      }//sections
      Objects->AddObject(pObject);
      delete pObject;
    }//read mesh
  }//readed correctly


  InFile.Close();
  if (!InFile.Open(CString("..\\Bound\\")+RealFileName+CString("_BOUND.bnd"), CFile::modeRead))
  {
    AfxGetApp()->DoWaitCursor(0);
    ShowFailMessage(  pwnd, CString(CString("Error opening ")+CString("..\\Bound\\")+RealFileName+CString("_BOUND.bnd file")).GetBuffer(256),
              "file missing",
              MB_ICONEXCLAMATION);
    return 1;
  }

  /////////////////////
  // Read the boundary;
  char buffer[256];
  int verts=-1;
  int faces=-1;
  float fval;
  ReadName(&InFile, (char*)&buffer);
  while (!CharMatch((char*)&buffer, "v"))
  {//read header;
    if (CharMatch((char*)&buffer, "version:")) fval = ReadFloat(&InFile);
    if (CharMatch((char*)&buffer, "verts:")) verts = ReadLong(&InFile);
    if (CharMatch((char*)&buffer, "polys:")) faces = ReadLong(&InFile);
    ReadName(&InFile, (char*)&buffer);
  }//read header;
  if ((verts <=0) || (faces <= 0))
  {
    AfxGetApp()->DoWaitCursor(0);
    ShowFailMessage(  pwnd, "Invalid _BOUND.bnd file.", "\"verts:\" or \"polys:\" keyword missing",
              MB_ICONEXCLAMATION);
    return 1;
  }

  tObject*  pBound = new tObject;
  strcpy(pBound->ObjectName, "BOUND");
  pBound->VertTable->VertAmount = verts;
  pBound->VertTable->Table = new tNormalVertex[verts];
  for (int v = 0; v < verts; v++)
  {
    pBound->VertTable->Table[v].X = -ReadFloat(&InFile);
    pBound->VertTable->Table[v].Y = ReadFloat(&InFile);
    pBound->VertTable->Table[v].Z = -ReadFloat(&InFile);
    pBound->VertTable->Table[v].NormalX = pBound->VertTable->Table[v].X;
    pBound->VertTable->Table[v].NormalY = pBound->VertTable->Table[v].Y;
    pBound->VertTable->Table[v].NormalZ = pBound->VertTable->Table[v].Z;
    pBound->VertTable->Table[v].NormalizeNormal();
    ReadName(&InFile, (char*)&buffer); //"v"
  }

  int i1,i2,i3,i4, fc=0;
  while (!EndOfFile(&InFile) && (fc<faces))
  {
    while (!CharMatch((char*)&buffer, "tri") && !CharMatch((char*)&buffer, "quad"))
      ReadName(&InFile, (char*)&buffer); // skip
    if (CharMatch((char*)&buffer, "tri"))
    {//read and add face;
      i1 = ReadLong(&InFile);
      i2 = ReadLong(&InFile);
      i3 = ReadLong(&InFile);
      pBound->FaceTable->AddFace(i2,i1,i3);
    }//read and add face;
    if (CharMatch((char*)&buffer, "quad"))
    {//read and add quad;
      i1 = ReadLong(&InFile);
      i2 = ReadLong(&InFile);
      i3 = ReadLong(&InFile);
      i4 = ReadLong(&InFile);
      pBound->FaceTable->AddFace(i2,i1,i3);
      pBound->FaceTable->AddFace(i3,i1,i4);
    }//read and add quad;
    fc++;
    ReadName(&InFile, (char*)&buffer); //next;
  }
  InFile.Close();

  //// add boundary;
  Objects->AddObject(pBound);
  delete pBound;

  AfxGetApp()->DoWaitCursor(0);

  return 1;
}


/**********************************************************************/
//    MM2 PKG File Exporter for Zanoza 3D Editor
/**********************************************************************/
DWORD CALLBACK Export(  CString Tofile,
            CWnd* pwnd,
            tObjectSet* Objects,
            tUnDataSet* UnData,
            CDirect3D*  d3d,
            SYSTEMREQUESTPROC RequestProc,
            HINSTANCE AppHIns,
            HINSTANCE DllHIns)
{
  CFile OutFile;
  FILE* filemtx;
  
  if (!OutFile.Open(Tofile, CFile::modeWrite | CFile::modeCreate))
    return ShowFailMessage(  pwnd, "Error opening file for exporting",
                "already opened, or has invalid attributes",
                MB_OK | MB_ICONEXCLAMATION);



  AfxSetResourceHandle(DllHIns);
  CExportDialog dlg(pwnd);
  if (dlg.DoModal() != IDOK) return 0;
  AfxSetResourceHandle(AppHIns);


  AfxGetApp()->DoWaitCursor(1);

  long  NumInSequence = 0;
  CString  ObjSequence[55];
  BOOL  *bUsed = NULL;

  CString  Suffixes[4];
  Suffixes[0] = "_H";
  Suffixes[1] = "_M";
  Suffixes[2] = "_L";
  Suffixes[3] = "_VL";
  
  if (dlg.Choice == 1)//dashboard
  {
    ObjSequence[0] = "DAMAGE_NEEDLE";
    ObjSequence[1] = "DASH";
    ObjSequence[2] = "GEAR_INDICATOR";
    ObjSequence[3] = "ROOF";
    ObjSequence[4] = "SPEED_NEEDLE";
    ObjSequence[5] = "TACH_NEEDLE";
    ObjSequence[6] = "WHEEL";
    NumInSequence = 7;
  }
  else
  {
    ObjSequence[0] = "BODY";
    ObjSequence[1] = "SHADOW";
    ObjSequence[2] = "HLIGHT";
    ObjSequence[3] = "TLIGHT";
    ObjSequence[4] = "RLIGHT";
    ObjSequence[5] = "BLIGHT";
    ObjSequence[6] = "SLIGHT0";
    ObjSequence[7] = "SLIGHT1";
    ObjSequence[8] = "SIREN0";
    ObjSequence[9] = "SIREN1";
    ObjSequence[10]= "SIREN2";
    ObjSequence[11]= "SIREN3";
    ObjSequence[12]= "SIREN4";
    ObjSequence[13]= "SIREN5";
    ObjSequence[14]= "WHL0";
    ObjSequence[15]= "WHL1";
    ObjSequence[16]= "WHL2";
    ObjSequence[17]= "WHL3";
    ObjSequence[18]= "BREAK0";
    ObjSequence[19]= "BREAK1";
    ObjSequence[20]= "BREAK2";
    ObjSequence[21]= "BREAK3";
    ObjSequence[22]= "BREAK01";
    ObjSequence[23]= "BREAK12";
    ObjSequence[24]= "BREAK23";
    ObjSequence[25]= "BREAK03";
    ObjSequence[26]= "TRAILER_HITCH";
    ObjSequence[27]= "SRN0";
    ObjSequence[28]= "SRN1";
    ObjSequence[29]= "SRN2";
    ObjSequence[30]= "SRN3";
    ObjSequence[31]= "SRN4";
    ObjSequence[32]= "SRN5";
    ObjSequence[33]= "HEADLIGHT0";
    ObjSequence[34]= "HEADLIGHT1";
    ObjSequence[35]= "EXHAUST0";
    ObjSequence[36]= "EXHAUST1";
    ObjSequence[37]= "WHL4";
    ObjSequence[38]= "WHL5";
    ObjSequence[39]= "WHL6";
    ObjSequence[40]= "WHL7";
    ObjSequence[41]= "WHL8";
    ObjSequence[42]= "WHL9";
    ObjSequence[43]= "FNDR0";
    ObjSequence[44]= "FNDR1";
    ObjSequence[45]= "FNDR2";
    ObjSequence[46]= "FNDR3";
    ObjSequence[47]= "FNDR4";
    ObjSequence[48]= "FNDR5";
    ObjSequence[49]= "BOUND";
    ObjSequence[50]= "H";
    ObjSequence[51]= "M";
    ObjSequence[52]= "L";
    ObjSequence[53]= "VL";
    ObjSequence[54]= "";//any other name
    NumInSequence = 55;
    if (dlg.Choice == 2)//trailer
    {
      ObjSequence[0] = "TRAILER";
      ObjSequence[15]= "TWHL0";
      ObjSequence[16]= "TWHL1";
      ObjSequence[17]= "TWHL2";
      ObjSequence[18]= "TWHL3";
      ObjSequence[37]= "TWHL4";
      ObjSequence[38]= "TWHL5";
      ObjSequence[39]= "TWHL6";
      ObjSequence[40]= "TWHL7";
      ObjSequence[41]= "TWHL8";
      ObjSequence[42]= "TWHL9";
    }
  }


  long  ObjCount, FaceCount;
  long  TotalVerts;
  long  TotalFaces;
  long  Written;
  long  IIndex;
  int    i;
  sFileDescriptor  FileDescriptor;
  sSection    Section;
  long  SectionFaces;
  tPOINT  CenterPoint;
  tObject*  pObj;
  short  i1;
  tList  MaterialsList;
  tList  GlobalMaterialsList;
  tList  FaceIndexList;

  char  cID[5];    cID[4] = 0;
  char  cFILE[5];  cFILE[4] = 0;
  BYTE  Zero = 0;
  BYTE  Len = 0;
  strcpy(cID, "PKG3");
  strcpy(cFILE, "FILE");
  OutFile.Write(cID, 4);
  BYTE  MaterialsMap[256];


  CString RealFileName = Tofile.Left(Tofile.GetLength()-4); // - ".pkg"
  int slashpos = RealFileName.ReverseFind('\\');
  if (slashpos == -1) slashpos = RealFileName.ReverseFind('/');
  if (slashpos != -1)
    RealFileName = RealFileName.Right(RealFileName.GetLength()-slashpos-1);

  filemtx = fopen(RealFileName+CString(".txt"), "w");
  fputs("-----------------------------------------------------------\n"
        "  This is a text-copy of all MTX files, that were created\n"
        " during exporting of PKG file.\n"
        "  MM2 doesn't require this file. It can only be used for\n"
        " copying values into Tuning files.\n"
        "-----------------------------------------------------------\n",
        filemtx);

  bUsed = new BOOL[Objects->ObjAmount];
  for (i = 0; i < Objects->ObjAmount; i++)
    bUsed[i] = FALSE;


CString UpperName;
for (int namecount = 0; namecount < NumInSequence; namecount++)
for (int suffcount = 0; suffcount < 4; suffcount++)
for (ObjCount = 0; ObjCount < Objects->ObjAmount; ObjCount++)
{
  UpperName = CString(Objects->ObjSet[ObjCount].ObjectName);
  UpperName.MakeUpper();
  if ( ((namecount < 49) &&
      (UpperName == ObjSequence[namecount] + Suffixes[suffcount]) ||
      (UpperName == ObjSequence[namecount] + Suffixes[suffcount] + CString(":M")))
    ||
     ((namecount >= 49) &&
      !bUsed[ObjCount] && (suffcount == 0)))
  {//this object
    if ((CString(Objects->ObjSet[ObjCount].ObjectName) != CString("BOUND")) && 
      (CString(Objects->ObjSet[ObjCount].ObjectName) != CString("bound")) &&
      (CString(Objects->ObjSet[ObjCount].ObjectName) != CString("UVMapperDATA")))
    {//exporting objects;
      OutFile.Write(cFILE, 4);
    
      pObj = &Objects->ObjSet[ObjCount];
      CString name = CString(Objects->ObjSet[ObjCount].ObjectName);
      name.MakeLower();
      int pos = name.Find(":m");
      if ( pos >=0 )  Len = pos+1;
      else      Len = CString(Objects->ObjSet[ObjCount].ObjectName).GetLength()+1;
      OutFile.Write(&Len, 1);
      OutFile.Write(Objects->ObjSet[ObjCount].ObjectName, Len-1);
      OutFile.Write(&Zero, 1);
      Written = 0;
      FileDescriptor.lSections1 = 0;
      FileDescriptor.lSections2 = 0;
      FileDescriptor.lTotalFaces = 0;
      FileDescriptor.lTotalVerts = 0;
      FileDescriptor.nFlags = 0x112;
      OutFile.Write(&Written, 4);
      OutFile.Write(&FileDescriptor, sizeof(sFileDescriptor));
      Written += 4+sizeof(sFileDescriptor);
    
      tVert  Vert;
      TotalVerts = 0;
      TotalFaces = 0;
      Section.lUnknown1 = 1;
      Section.lUnknown2 = 3;

      if (pos >= 0)
      {//create mtx file;
        CenterPoint = GetMatrixPosition(pObj->mTransform);
        tPOINT max, min;
        pObj->GetMaxMinPoints(&max, &min);
        int pos2 = name.Find("_h");
        if (pos2 == -1) pos2 = name.Find("_m");
        if (pos2 == -1) pos2 = name.Find("_l");
        if (pos2 == -1) pos2 = name.Find("_vl");
        if (pos2 >= 0)  name = name.Left(pos2);
        else      name = name.Left(pos);
        name += CString(".mtx");
        name = RealFileName+CString("_")+name;
        float fval;
        tPOINT center;
        center.x=0;center.y=0;center.z=0;
        for (int VertCount=0; VertCount < pObj->VertTable->VertAmount; VertCount++)
        {
          center.x+=pObj->VertTable->Table[VertCount].X/(float)pObj->VertTable->VertAmount;
          center.y+=pObj->VertTable->Table[VertCount].Y/(float)pObj->VertTable->VertAmount;
          center.z+=pObj->VertTable->Table[VertCount].Z/(float)pObj->VertTable->VertAmount;
        }
        if ((suffcount==0) && MiscFile.Open(name, CFile::modeCreate | CFile::modeWrite))
        {//write;
          fval= min.x-CenterPoint.x; MiscFile.Write(&fval, 4);
          fval= min.y-CenterPoint.y; MiscFile.Write(&fval, 4);
          fval= min.z-CenterPoint.z; MiscFile.Write(&fval, 4);
          fval= max.x-CenterPoint.x; MiscFile.Write(&fval, 4);
          fval= max.y-CenterPoint.y; MiscFile.Write(&fval, 4);
          fval= max.z-CenterPoint.z; MiscFile.Write(&fval, 4);
          fval= center.x;      MiscFile.Write(&fval, 4);
          fval= center.y;      MiscFile.Write(&fval, 4); 
          fval= center.z;      MiscFile.Write(&fval, 4);
          fval= -CenterPoint.x; MiscFile.Write(&fval, 4);
          fval= CenterPoint.y; MiscFile.Write(&fval, 4);
          fval= -CenterPoint.z; MiscFile.Write(&fval, 4);
          MiscFile.Close();
          fprintf(filemtx,
            "Object: %s\n"
            "{\n"
            "  Center: {%f, %f, %f}\n"
            "  Center of mass: (%f, %f, %f}\n"
            "  Bounds: {%f, %f, %f}\n"
            "}\n",
            Objects->ObjSet[ObjCount].ObjectName,
            -CenterPoint.x, CenterPoint.y, -CenterPoint.z,
            center.x, center.y, -center.z,//negative x and z?
            max.x-CenterPoint.x, max.y-CenterPoint.y, max.z-CenterPoint.z);
        }  
      }//create mtx file;
      else
        CenterPoint = tPOINT(0.0f, 0.0f, 0.0f);

      for (FaceCount = 0; FaceCount < pObj->FaceTable->FaceAmount; FaceCount++)
      {
        if (!MaterialsList.Find(pObj->FaceTable->Table[FaceCount].Material))
          MaterialsList.Add(pObj->FaceTable->Table[FaceCount].Material);
        if (!GlobalMaterialsList.Find(pObj->FaceTable->Table[FaceCount].Material))
        {
          MaterialsMap[pObj->FaceTable->Table[FaceCount].Material] = (BYTE)GlobalMaterialsList.Amount;
          GlobalMaterialsList.Add(pObj->FaceTable->Table[FaceCount].Material);
        }
      }
      tItem*  pMatItem = MaterialsList.Head;
      for (long materials = 0; materials < MaterialsList.Amount; materials++)
      {
        VertexList*  pVertList = new VertexList;
        Section.lNumVertices  = 0;
        SectionFaces      = 0;
        Section.Material    = MaterialsMap[pMatItem->value];
        for (FaceCount = 0; FaceCount < pObj->FaceTable->FaceAmount; FaceCount++)
          if (pObj->FaceTable->Table[FaceCount].Material == pMatItem->value)
          {
            Vert.vert.x = -pObj->VertTable->Table[pObj->FaceTable->Table[FaceCount].I2].X+CenterPoint.x;
            Vert.vert.y = pObj->VertTable->Table[pObj->FaceTable->Table[FaceCount].I2].Y-CenterPoint.y;
            Vert.vert.z = -pObj->VertTable->Table[pObj->FaceTable->Table[FaceCount].I2].Z+CenterPoint.z;
            Vert.vert.nx = -pObj->VertTable->Table[pObj->FaceTable->Table[FaceCount].I2].NormalX;
            Vert.vert.ny = pObj->VertTable->Table[pObj->FaceTable->Table[FaceCount].I2].NormalY;
            Vert.vert.nz = -pObj->VertTable->Table[pObj->FaceTable->Table[FaceCount].I2].NormalZ;
            Vert.vert.u = pObj->FaceTable->Table[FaceCount].U2;
            Vert.vert.v = pObj->FaceTable->Table[FaceCount].V2;
            IIndex = pVertList->Find(Vert);
            if (IIndex == -1)
            {
              FaceIndexList.Add(pVertList->Add(Vert));
              TotalVerts++;
              Section.lNumVertices++;
            }
            else
              FaceIndexList.Add(IIndex);
            Vert.vert.x = -pObj->VertTable->Table[pObj->FaceTable->Table[FaceCount].I1].X+CenterPoint.x;
            Vert.vert.y = pObj->VertTable->Table[pObj->FaceTable->Table[FaceCount].I1].Y-CenterPoint.y;
            Vert.vert.z = -pObj->VertTable->Table[pObj->FaceTable->Table[FaceCount].I1].Z+CenterPoint.z;
            Vert.vert.nx = -pObj->VertTable->Table[pObj->FaceTable->Table[FaceCount].I1].NormalX;
            Vert.vert.ny = pObj->VertTable->Table[pObj->FaceTable->Table[FaceCount].I1].NormalY;
            Vert.vert.nz = -pObj->VertTable->Table[pObj->FaceTable->Table[FaceCount].I1].NormalZ;
            Vert.vert.u = pObj->FaceTable->Table[FaceCount].U1;
            Vert.vert.v = pObj->FaceTable->Table[FaceCount].V1;
            IIndex = pVertList->Find(Vert);
            if (IIndex == -1)
            {
              FaceIndexList.Add(pVertList->Add(Vert));
              TotalVerts++;
              Section.lNumVertices++;
            }
            else
              FaceIndexList.Add(IIndex);
            Vert.vert.x = -pObj->VertTable->Table[pObj->FaceTable->Table[FaceCount].I3].X+CenterPoint.x;
            Vert.vert.y = pObj->VertTable->Table[pObj->FaceTable->Table[FaceCount].I3].Y-CenterPoint.y;
            Vert.vert.z = -pObj->VertTable->Table[pObj->FaceTable->Table[FaceCount].I3].Z+CenterPoint.z;
            Vert.vert.nx = -pObj->VertTable->Table[pObj->FaceTable->Table[FaceCount].I3].NormalX;
            Vert.vert.ny = pObj->VertTable->Table[pObj->FaceTable->Table[FaceCount].I3].NormalY;
            Vert.vert.nz = -pObj->VertTable->Table[pObj->FaceTable->Table[FaceCount].I3].NormalZ;
            Vert.vert.u = pObj->FaceTable->Table[FaceCount].U3;
            Vert.vert.v = pObj->FaceTable->Table[FaceCount].V3;
            IIndex = pVertList->Find(Vert);
            if (IIndex == -1)
            {
              FaceIndexList.Add(pVertList->Add(Vert));
              TotalVerts++;
              Section.lNumVertices++;
            }
            else
              FaceIndexList.Add(IIndex);
            if (!MaterialsList.Find(pObj->FaceTable->Table[FaceCount].Material))
              MaterialsList.Add(pObj->FaceTable->Table[FaceCount].Material);
            TotalFaces++;
            SectionFaces+=3;
          }
        pMatItem = pMatItem->pNext;
        OutFile.Write(&Section, sizeof(Section));
        Written += sizeof(Section) + sizeof(sVertex)*Section.lNumVertices;
      
        VertListNode*  vPtr = pVertList->Head;
        for (i = 0; i < Section.lNumVertices; i++)
        {//write vertices;
          OutFile.Write(&vPtr->vert.vert, sizeof(sVertex));
          vPtr = vPtr->next;
        }//write vertices;
      
        //write faces;
        OutFile.Write(&SectionFaces, 4);
        Written += 4 + SectionFaces*2;
        tItem*  pIndex = FaceIndexList.Head;
        while (pIndex)
        {//write this face indexes;
          i1 = (short)pIndex->value;  OutFile.Write(&i1, 2); pIndex = pIndex->pNext;//i1
          i1 = (short)pIndex->value;  OutFile.Write(&i1, 2); pIndex = pIndex->pNext;//i2
          i1 = (short)pIndex->value;  OutFile.Write(&i1, 2); pIndex = pIndex->pNext;//i3
        }//write this face indexes;
        FaceIndexList.Delete();
        delete pVertList;
      }//section (by material)

      OutFile.Seek(-Written, CFile::current);
      Written-=4;
      OutFile.Write(&Written, 4);//size
      FileDescriptor.lSections1 = MaterialsList.Amount;
      FileDescriptor.lSections2 = MaterialsList.Amount;
      FileDescriptor.lTotalFaces= TotalFaces*3;
      FileDescriptor.lTotalVerts= TotalVerts;
      OutFile.Write(&FileDescriptor, sizeof(FileDescriptor));
      OutFile.Seek(Written - sizeof(FileDescriptor), CFile::current);

      MaterialsList.Delete();
    }//exporting objects;
    else
      if (((CString(Objects->ObjSet[ObjCount].ObjectName) != CString("UVMapperDATA"))) &&
        MiscFile.Open(CString("..\\bound\\")+RealFileName+CString("_BOUND.bnd"), CFile::modeCreate | CFile::modeWrite))
      {//export boundary
        pObj = &Objects->ObjSet[ObjCount];
        WriteName (&MiscFile, "version: 1.01\n");
        WriteLongWithName (&MiscFile, "verts: ", pObj->VertTable->VertAmount, TRUE);
        WriteName (&MiscFile, "materials: 1\nedges: 0\n");
        WriteLongWithName (&MiscFile, "polys: ", pObj->FaceTable->FaceAmount, TRUE);
        for (int vcount = 0; vcount < pObj->VertTable->VertAmount; vcount++)
        {
          WriteName (&MiscFile, "\nv\t");
          WriteFloat (&MiscFile, -pObj->VertTable->Table[vcount].X, FALSE);
          WriteName (&MiscFile, "\t");
          WriteFloat (&MiscFile, pObj->VertTable->Table[vcount].Y, FALSE);
          WriteName (&MiscFile, "\t");
          WriteFloat (&MiscFile, -pObj->VertTable->Table[vcount].Z, FALSE);
        }
        WriteName (&MiscFile, "\n\nmtl default {\n\telasticity: 0.100000\n\tfriction: 0.500000\n\teffect: none\n\tsound: 0\n}\n");

        for (int fcount = 0; fcount < pObj->FaceTable->FaceAmount; fcount++)
        {
          WriteName (&MiscFile, "\ntri\t");
          WriteLong (&MiscFile, pObj->FaceTable->Table[fcount].I2, FALSE);WriteName (&MiscFile, "   ");
          WriteLong (&MiscFile, pObj->FaceTable->Table[fcount].I1, FALSE);WriteName (&MiscFile, "   ");
          WriteLong (&MiscFile, pObj->FaceTable->Table[fcount].I3, FALSE);WriteName (&MiscFile, "   ");
          WriteLong (&MiscFile, 0, FALSE);
        }
        MiscFile.Close();
      }//export boundary
    bUsed[ObjCount] = TRUE;//this object will not be exported twice.
  }//this object
}//objects sequence

  if (bUsed) delete[] bUsed;
  //=======================
  //Export Materials;
  OutFile.Write(cFILE, 4);
  Len = 8; OutFile.Write(&Len, 1);
  OutFile.Write("shaders", 7); OutFile.Write(&Zero, 1);

  long MatWritten = 4;
  OutFile.Write(&MatWritten, 4); //size;
  DWORD MatType;
  long MatAmount = GlobalMaterialsList.Amount;
  CString tofile = Tofile; tofile.MakeLower();
  CString PrevTexture = "";
  CString CurrTexture = "";
  tItem  *pCurMat;

  if (tofile.Find("_dash") >=0)  // dash materials
    MatType = 0x09;
  else
    MatType = 0x80 | (dlg.NumPaintjobs+1);//0x84; //may be this is num of paintjobs???
  MatWritten +=8;
  OutFile.Write(&MatType, 4); OutFile.Write(&MatAmount, 4);


  for (int paint = 0; paint < dlg.NumPaintjobs+1; paint++)
  {//material
    if (tofile.Find("_dash") >=0)
    {// dash materials
      pCurMat = GlobalMaterialsList.Head;
      sDashMaterial  DashMaterial;
      for (int materials = 0; materials < MatAmount; materials++)
      {
        if (d3d->Materials.Materials[pCurMat->value].MatParams.PrimTexture >=0)
        {
          CurrTexture = CString(d3d->Materials.Textures->GetName(d3d->Materials.Materials[pCurMat->value].MatParams.PrimTexture));
          CurrTexture = CurrTexture.Left(CurrTexture.ReverseFind('.'));
          if (paint > 0)
          {//may be replace???
            int posstart = CurrTexture.Find(dlg.DefSuffix);
            if (posstart > 0)
            {//replace suffix
              int posend = posstart+dlg.DefSuffix.GetLength();
              CurrTexture = CurrTexture.Left(posstart)+
                    dlg.Suffixes[paint-1]+
                    CurrTexture.Right(CurrTexture.GetLength()-posend);
            }//replace suffix
          }
          if (PrevTexture != CurrTexture)
          {//write texture name;
            PrevTexture = CurrTexture;
            Len = PrevTexture.GetLength()+1;
            OutFile.Write(&Len, 1);
            OutFile.Write(PrevTexture.GetBuffer(Len), Len);
            MatWritten += Len+1;
          }//write texture name;
          else
          {
            PrevTexture = "";
            OutFile.Write(&Zero, 1);
            MatWritten ++;
          }
        }
        else
        {
          PrevTexture = "";
          OutFile.Write(&Zero, 1);
          MatWritten ++;
        }
        DashMaterial.rgba_ambient.r = d3d->Materials.Materials[pCurMat->value].MatRec.ambient.r;
        DashMaterial.rgba_ambient.g = d3d->Materials.Materials[pCurMat->value].MatRec.ambient.g;
        DashMaterial.rgba_ambient.b = d3d->Materials.Materials[pCurMat->value].MatRec.ambient.b;
        DashMaterial.rgba_ambient.a = d3d->Materials.Materials[pCurMat->value].MatRec.ambient.a;

        DashMaterial.rgba_diffuse.r = d3d->Materials.Materials[pCurMat->value].MatRec.diffuse.r;
        DashMaterial.rgba_diffuse.g = d3d->Materials.Materials[pCurMat->value].MatRec.diffuse.g;
        DashMaterial.rgba_diffuse.b = d3d->Materials.Materials[pCurMat->value].MatRec.diffuse.b;
        DashMaterial.rgba_diffuse.a = d3d->Materials.Materials[pCurMat->value].MatRec.diffuse.a;

        DashMaterial.rgba_specular.r = d3d->Materials.Materials[pCurMat->value].MatRec.specular.r;
        DashMaterial.rgba_specular.g = d3d->Materials.Materials[pCurMat->value].MatRec.specular.g;
        DashMaterial.rgba_specular.b = d3d->Materials.Materials[pCurMat->value].MatRec.specular.b;
        DashMaterial.rgba_specular.a = d3d->Materials.Materials[pCurMat->value].MatRec.specular.a;

        DashMaterial.rgba_emissive.r = 0.1f;
        DashMaterial.rgba_emissive.g = 0.1f;
        DashMaterial.rgba_emissive.b = 0.1f;
        DashMaterial.rgba_emissive.a = 1.0f;

        DashMaterial.power = -1.0f;//d3d->Materials.Materials[pCurMat->value].MatRec.power;
        if (d3d->Materials.Materials[pCurMat->value].MatParams.PrimApply == D3DTOP_ADD)
        {
          DashMaterial.rgba_ambient.a = 0.0f;
          DashMaterial.rgba_diffuse.a = 0.0f;
          DashMaterial.rgba_specular.a = 0.0f;
          DashMaterial.rgba_emissive.a = 0.0f;
        }
        OutFile.Write(&DashMaterial, sizeof(sDashMaterial));
        MatWritten += sizeof(sDashMaterial);
        pCurMat = pCurMat->pNext;
      }
    }
    else
    {
      pCurMat = GlobalMaterialsList.Head;
      sMaterial  Material;
      for (int materials = 0; materials < MatAmount; materials++)
      {
        if (d3d->Materials.Materials[pCurMat->value].MatParams.PrimTexture >=0)
        {
          CurrTexture = CString(d3d->Materials.Textures->GetName(d3d->Materials.Materials[pCurMat->value].MatParams.PrimTexture));
          CurrTexture = CurrTexture.Left(CurrTexture.ReverseFind('.'));
          if (paint > 0)
          {//may be replace???
            int posstart = CurrTexture.Find(dlg.DefSuffix);
            if (posstart > 0)
            {//replace suffix
              int posend = posstart+dlg.DefSuffix.GetLength();
              CurrTexture = CurrTexture.Left(posstart)+
                    dlg.Suffixes[paint-1]+
                    CurrTexture.Right(CurrTexture.GetLength()-posend);
            }//replace suffix
          }

          if (PrevTexture != CurrTexture)
          {//write texture name;
            PrevTexture = CurrTexture;
            Len = PrevTexture.GetLength()+1;
            OutFile.Write(&Len, 1);
            OutFile.Write(PrevTexture.GetBuffer(Len), Len);
            MatWritten += Len+1;
          }//write texture name;
          else
          {
            PrevTexture = "";
            OutFile.Write(&Zero, 1);
            MatWritten ++;
          }
        }
        else
        {
          PrevTexture = "";
          OutFile.Write(&Zero, 1);
          MatWritten ++;
        }

        Material.rgba_ambient.r = (BYTE)(d3d->Materials.Materials[pCurMat->value].MatRec.ambient.r*255);
        Material.rgba_ambient.g = (BYTE)(d3d->Materials.Materials[pCurMat->value].MatRec.ambient.g*255);
        Material.rgba_ambient.b = (BYTE)(d3d->Materials.Materials[pCurMat->value].MatRec.ambient.b*255);
        Material.rgba_ambient.a = (BYTE)(d3d->Materials.Materials[pCurMat->value].MatRec.ambient.a*255);

        Material.rgba_diffuse.r = (BYTE)(d3d->Materials.Materials[pCurMat->value].MatRec.diffuse.r*255);
        Material.rgba_diffuse.g = (BYTE)(d3d->Materials.Materials[pCurMat->value].MatRec.diffuse.g*255);
        Material.rgba_diffuse.b = (BYTE)(d3d->Materials.Materials[pCurMat->value].MatRec.diffuse.b*255);
        Material.rgba_diffuse.a = (BYTE)(d3d->Materials.Materials[pCurMat->value].MatRec.diffuse.a*255);

        Material.rgba_specular.r = 0;
        Material.rgba_specular.g = 0;
        Material.rgba_specular.b = 0;
        Material.rgba_specular.a = 0;

        Material.power = 0.5f;
        if (d3d->Materials.Materials[pCurMat->value].MatParams.PrimApply == D3DTOP_ADD)
        {
          Material.rgba_ambient.a = 0;
          Material.rgba_diffuse.a = 0;
          Material.rgba_specular.a = 0;
        }
        OutFile.Write(&Material, sizeof(sMaterial));
        MatWritten += sizeof(Material);
        pCurMat = pCurMat->pNext;
      }
    }
  }//paintjobs
  OutFile.Seek(-MatWritten, CFile::current);
  MatWritten-=4;
  OutFile.Write(&MatWritten, 4);//size
  OutFile.Seek(MatWritten, CFile::current);

  GlobalMaterialsList.Delete();

  //=================================
  //  offset;
  OutFile.Write(cFILE, 4);
  Len = 7;
  OutFile.Write(&Len, 1); OutFile.Write("offset", 6);OutFile.Write(&Zero, 1);
  MatWritten = 12;
  float fZero = 0.0f;
  OutFile.Write(&MatWritten, 4);
  OutFile.Write(&fZero, 4); OutFile.Write(&fZero, 4); OutFile.Write(&fZero, 4);


  OutFile.Close();
  fclose(filemtx);

  AfxGetApp()->DoWaitCursor(0);

  return 0;
}
